import { DmnDocumentData } from "./DmnDocumentData";
import { DmnDecision } from "./DmnDecision";
import * as path from "path";
import { getMarshaller } from "@kie-tools/dmn-marshaller";
import { ns as dmn15ns } from "@kie-tools/dmn-marshaller/dist/schemas/dmn-1_5/ts-gen/meta";
import { DMN15_SPEC } from "@kie-tools/dmn-marshaller/dist/schemas/dmn-1_5/Dmn15Spec";
import { v4 as uuid } from "uuid";
const INPUT_DATA = "inputData";
const XML_MIME = "text/xml";
const DECISION_NAME_ATTRIBUTE = "name";
const NAMESPACE = "namespace";
const DMN_NAME = "name";
const DECISION = "decision";
const DEFINITIONS = "definitions";
const generateUuid = () => {
    return `_${uuid()}`.toLocaleUpperCase();
};
const EMPTY_DMN = () => `<?xml version="1.0" encoding="UTF-8"?>
<definitions
  xmlns="${dmn15ns.get("")}"
  expressionLanguage="${DMN15_SPEC.expressionLanguage.default}"
  namespace="https://kie.org/dmn/${generateUuid()}"
  id="${generateUuid()}"
  name="DMN${generateUuid()}">
</definitions>`;
export class DmnLanguageService {
    constructor(args) {
        this.args = args;
        this.parser = new DOMParser();
        this.inputDataRegEx = new RegExp(`([a-z]*:)?(${INPUT_DATA})`);
        this.decisionsTagRegExp = new RegExp(`([a-z]*:)?(${DECISION})`);
        this.definitionsTagRegExp = new RegExp(`([a-z]*:)?(${DEFINITIONS})`);
    }
    async buildImportIndexModel(normalizedPosixPathRelativeToTheWorkspaceRoot) {
        const xml = await this.args.getModelXml({ normalizedPosixPathRelativeToTheWorkspaceRoot });
        return {
            definitions: getMarshaller(xml, { upgradeTo: "latest" }).parser.parse().definitions,
            xml,
        };
    }
    async recusivelyPopulateImportIndex(normalizedPosixPathRelativeToTheWorkspaceRoot, importIndex, parents, depth) {
        var _a, _b, _c, _d, _e, _f;
        if (depth === 0) {
            return;
        }
        const model = (_a = importIndex.models.get(normalizedPosixPathRelativeToTheWorkspaceRoot)) !== null && _a !== void 0 ? _a : importIndex.models
            .set(normalizedPosixPathRelativeToTheWorkspaceRoot, await this.buildImportIndexModel(normalizedPosixPathRelativeToTheWorkspaceRoot))
            .get(normalizedPosixPathRelativeToTheWorkspaceRoot);
        const hierarchy = (_b = importIndex.hierarchy.get(normalizedPosixPathRelativeToTheWorkspaceRoot)) !== null && _b !== void 0 ? _b : importIndex.hierarchy
            .set(normalizedPosixPathRelativeToTheWorkspaceRoot, { immediate: new Set(), deep: new Set() })
            .get(normalizedPosixPathRelativeToTheWorkspaceRoot);
        const basedir = path.dirname(normalizedPosixPathRelativeToTheWorkspaceRoot);
        for (const i of (_c = model.definitions.import) !== null && _c !== void 0 ? _c : []) {
            const locationUri = i["@_locationURI"];
            if (!locationUri) {
                console.warn(`Ignoring import with namespace '${i["@_namespace"]}' because it doesn't have a locationURI.`);
                continue;
            }
            const p = path.posix.join(basedir, path.posix.normalize(locationUri));
            if (hierarchy.immediate.has(p)) {
                for (const parent of parents) {
                    const parentHierarchy = importIndex.hierarchy.get(parent);
                    parentHierarchy === null || parentHierarchy === void 0 ? void 0 : parentHierarchy.deep.add(p);
                    for (const i of (_e = (_d = importIndex.hierarchy.get(p)) === null || _d === void 0 ? void 0 : _d.deep) !== null && _e !== void 0 ? _e : []) {
                        parentHierarchy === null || parentHierarchy === void 0 ? void 0 : parentHierarchy.deep.add(i);
                    }
                }
                continue;
            }
            hierarchy.immediate.add(p);
            hierarchy.deep.add(p);
            for (const parent of parents) {
                (_f = importIndex.hierarchy.get(parent)) === null || _f === void 0 ? void 0 : _f.deep.add(p);
            }
            parents.push(normalizedPosixPathRelativeToTheWorkspaceRoot);
            await this.recusivelyPopulateImportIndex(p, importIndex, parents, depth - 1);
            parents.pop();
        }
    }
    async buildImportIndex(resources, depth = -1) {
        try {
            const importIndex = {
                hierarchy: new Map(),
                models: new Map(resources.map((r) => [
                    r.normalizedPosixPathRelativeToTheWorkspaceRoot,
                    {
                        xml: r.content,
                        definitions: getMarshaller(r.content || EMPTY_DMN(), { upgradeTo: "latest" }).parser.parse().definitions,
                    },
                ])),
            };
            for (const r of resources) {
                await this.recusivelyPopulateImportIndex(r.normalizedPosixPathRelativeToTheWorkspaceRoot, importIndex, [], depth);
            }
            return importIndex;
        }
        catch (error) {
            throw new Error(`
DMN LANGUAGE SERVICE - buildImportIndex: Error while getting imported models from model resources.
Tried to use the following model resources: ${JSON.stringify(resources)}
Error details: ${error}`);
        }
    }
    getPathFromNodeId(resources, nodeId) {
        var _a, _b, _c;
        for (const resourceContent of resources) {
            const xmlContent = this.parser.parseFromString((_a = resourceContent.content) !== null && _a !== void 0 ? _a : "", XML_MIME);
            const inputDataTag = this.inputDataRegEx.exec((_b = resourceContent.content) !== null && _b !== void 0 ? _b : "");
            const inputs = xmlContent.getElementsByTagName((_c = inputDataTag === null || inputDataTag === void 0 ? void 0 : inputDataTag[0]) !== null && _c !== void 0 ? _c : INPUT_DATA);
            for (const input of Array.from(inputs)) {
                if (input.id === nodeId) {
                    return resourceContent.normalizedPosixPathRelativeToTheWorkspaceRoot;
                }
            }
        }
        return "";
    }
    getDmnDocumentData(xml) {
        const xmlContent = this.parser.parseFromString(xml, XML_MIME);
        const definitionsTag = this.definitionsTagRegExp.exec(xml);
        const definitions = xmlContent.getElementsByTagName(definitionsTag ? definitionsTag[0] : DEFINITIONS);
        const definition = definitions[0];
        const namespace = definition.getAttribute(NAMESPACE);
        const dmnModelName = definition.getAttribute(DMN_NAME);
        const dmnDecisions = this.decisionsTagRegExp.exec(xml);
        const dmnDecisionsContent = xmlContent.getElementsByTagName(dmnDecisions ? dmnDecisions[0] : DECISION);
        const decisions = Array.from(dmnDecisionsContent)
            .map((decision) => decision.getAttribute(DECISION_NAME_ATTRIBUTE))
            .flatMap((decisionName) => (decisionName ? [new DmnDecision(decisionName)] : []));
        return new DmnDocumentData(namespace !== null && namespace !== void 0 ? namespace : "", dmnModelName !== null && dmnModelName !== void 0 ? dmnModelName : "", decisions);
    }
    getSpecVersion(xml) {
        if (xml === "") {
            return;
        }
        try {
            return getMarshaller(xml).originalVersion;
        }
        catch (error) {
            return;
        }
    }
}
//# sourceMappingURL=DmnLanguageService.js.map