import { OperatingSystem } from "@kie-tools-core/operating-system";
export var ModKeys;
(function (ModKeys) {
    ModKeys["CTRL"] = "ctrl";
    ModKeys["META"] = "meta";
    ModKeys["ALT"] = "alt";
    ModKeys["SHIFT"] = "shift";
})(ModKeys || (ModKeys = {}));
const MODIFIER_KEY_NAMES = new Map([
    ["AltLeft", "alt"],
    ["AltRight", "alt"],
    ["CtrlLeft", "ctrl"],
    ["CtrlRight", "ctrl"],
    ["MetaLeft", "meta"],
    ["MetaRight", "meta"],
    ["ShiftLeft", "shift"],
    ["ShiftRight", "shift"],
]);
const KEY_CODES = new Map([
    ["/", "Slash"],
    ["esc", "Escape"],
    ["delete", "Delete"],
    ["backspace", "Backspace"],
    ["space", "Space"],
    ["right", "ArrowRight"],
    ["left", "ArrowLeft"],
    ["up", "ArrowUp"],
    ["down", "ArrowDown"],
    ["a", "KeyA"],
    ["b", "KeyB"],
    ["c", "KeyC"],
    ["d", "KeyD"],
    ["e", "KeyE"],
    ["f", "KeyF"],
    ["g", "KeyG"],
    ["h", "KeyH"],
    ["i", "KeyI"],
    ["j", "KeyJ"],
    ["k", "KeyK"],
    ["l", "KeyL"],
    ["m", "KeyM"],
    ["n", "KeyN"],
    ["o", "KeyO"],
    ["p", "KeyP"],
    ["q", "KeyQ"],
    ["r", "KeyR"],
    ["s", "KeyS"],
    ["t", "KeyT"],
    ["u", "KeyU"],
    ["v", "KeyV"],
    ["w", "KeyW"],
    ["x", "KeyX"],
    ["y", "KeyY"],
    ["z", "KeyZ"],
]);
const IGNORED_TAGS = ["INPUT", "TEXTAREA", "SELECT", "OPTION"];
export class DefaultKeyboardShortcutsService {
    constructor(args) {
        this.args = args;
        this.eventIdentifiers = 1;
        this.keyBindings = new Map();
    }
    registerKeyDownThenUp(combination, label, onKeyDown, onKeyUp, opts) {
        console.debug(`Registering shortcut (down/up) for ${combination} - ${label}: ${opts === null || opts === void 0 ? void 0 : opts.repeat}`);
        const keyBinding = {
            combination,
            label,
            listener: (event) => {
                var _a;
                const keyboardEvent = getProcessableKeyboardEvent(combination, event, opts);
                if (!keyboardEvent) {
                    return true;
                }
                if (keyboardEvent.type === "keydown") {
                    if (setsEqual(this.combinationKeySet(combination), this.pressedKeySet(keyboardEvent))) {
                        console.debug(`Fired (down) [${combination}]!`);
                        onKeyDown(keyboardEvent.target);
                        return false;
                    }
                }
                else if (keyboardEvent.type === "keyup") {
                    if (this.combinationKeySet(combination).has((_a = MODIFIER_KEY_NAMES.get(keyboardEvent.code)) !== null && _a !== void 0 ? _a : "") ||
                        this.combinationKeySet(combination).has(keyboardEvent.code)) {
                        console.debug(`Fired (up) [${combination}]!`);
                        onKeyUp(keyboardEvent.target);
                        return false;
                    }
                }
                return true;
            },
            opts,
        };
        this.keyBindings.set(this.eventIdentifiers, keyBinding);
        this.keyBindingElement(keyBinding).addEventListener("keydown", keyBinding.listener);
        this.keyBindingElement(keyBinding).addEventListener("keyup", keyBinding.listener);
        return this.eventIdentifiers++;
    }
    registerKeyPress(combination, label, onKeyPress, opts) {
        console.debug(`Registering shortcut (press) for ${combination} - ${label}: ${opts === null || opts === void 0 ? void 0 : opts.repeat}`);
        const keyBinding = {
            combination,
            label,
            listener: (event) => {
                const keyboardEvent = getProcessableKeyboardEvent(combination, event, opts);
                if (!keyboardEvent) {
                    return true;
                }
                if (setsEqual(this.combinationKeySet(combination), this.pressedKeySet(keyboardEvent))) {
                    console.debug(`Fired (press) [${combination}]!`);
                    onKeyPress(keyboardEvent.target);
                    return false;
                }
                return true;
            },
            opts,
        };
        this.keyBindings.set(this.eventIdentifiers, keyBinding);
        this.keyBindingElement(keyBinding).addEventListener("keydown", keyBinding.listener);
        return this.eventIdentifiers++;
    }
    registerKeyPressOnce(combination, onKeyPress, opts) {
        const id = this.registerKeyPress(combination, "", async (target) => {
            onKeyPress(target);
            this.deregister(id);
        }, opts ? { ...opts, hidden: true } : opts);
        return id;
    }
    deregister(id) {
        const keyBinding = this.keyBindings.get(id);
        if (!keyBinding) {
            console.error(`Unable to de-register keyboard shortcut with id ${id} because it was not registered.`);
            return;
        }
        this.keyBindingElement(keyBinding).removeEventListener("keypress", keyBinding === null || keyBinding === void 0 ? void 0 : keyBinding.listener);
        this.keyBindingElement(keyBinding).removeEventListener("keydown", keyBinding === null || keyBinding === void 0 ? void 0 : keyBinding.listener);
        this.keyBindingElement(keyBinding).removeEventListener("keyup", keyBinding === null || keyBinding === void 0 ? void 0 : keyBinding.listener);
        this.keyBindings.delete(id);
    }
    keyBindingElement(keyBinding) {
        var _a, _b;
        return (_b = (_a = keyBinding === null || keyBinding === void 0 ? void 0 : keyBinding.opts) === null || _a === void 0 ? void 0 : _a.element) !== null && _b !== void 0 ? _b : window;
    }
    combinationKeySet(combination) {
        const keys = combination
            .split("+")
            .map((k) => k.toLowerCase())
            .map((k) => { var _a; return (_a = KEY_CODES.get(k)) !== null && _a !== void 0 ? _a : k; });
        if (this.args.os === OperatingSystem.MACOS) {
            return new Set(keys.map((k) => (k === ModKeys.CTRL ? ModKeys.META : k)));
        }
        else {
            return new Set(keys);
        }
    }
    pressedKeySet(e) {
        const pressedKeySet = new Set();
        if (e.ctrlKey) {
            pressedKeySet.add(ModKeys.CTRL);
        }
        if (e.metaKey) {
            pressedKeySet.add(ModKeys.META);
        }
        if (e.altKey) {
            pressedKeySet.add(ModKeys.ALT);
        }
        if (e.shiftKey) {
            pressedKeySet.add(ModKeys.SHIFT);
        }
        if (Array.from(MODIFIER_KEY_NAMES.keys()).indexOf(e.code) === -1) {
            pressedKeySet.add(e.code);
        }
        return pressedKeySet;
    }
    registered() {
        return Array.from(this.keyBindings.values());
    }
    isEnabled() {
        return true;
    }
}
function getProcessableKeyboardEvent(combination, event, opts) {
    if (event instanceof CustomEvent && IGNORED_TAGS.includes(event.detail.channelOriginalTargetTagName)) {
        console.debug(`Ignoring execution (${combination}) because target is ${event.detail.channelOriginalTargetTagName}`);
        return null;
    }
    const keyboardEvent = event instanceof CustomEvent ? new KeyboardEvent(event.detail.type, event.detail) : event;
    if (keyboardEvent.target instanceof Element && IGNORED_TAGS.includes(keyboardEvent.target.tagName)) {
        console.debug(`Ignoring execution (${combination}) because target is ${keyboardEvent.target.tagName}`);
        return null;
    }
    if (keyboardEvent.repeat && !(opts === null || opts === void 0 ? void 0 : opts.repeat)) {
        return null;
    }
    return keyboardEvent;
}
function setsEqual(lhs, rhs) {
    if (lhs.size !== rhs.size) {
        return false;
    }
    for (const a of lhs) {
        if (!rhs.has(a)) {
            return false;
        }
    }
    return true;
}
//# sourceMappingURL=DefaultKeyboardShortcutsService.js.map